%% Description
% Plots figure 15a and 15b.
% Fig 15a: examples of how the force changes as a function of the
% separation at different retraction speeds.
% Fig15b: the energy required to detach as a function of the retraction
% speed.

%% Parameter Definitions
%
% dH    Different retraction speeds used (fig 15a).
%
% H     Values of H_infty used in dynamic evolution (fig 15a).
%
% F     Adhesion force as calculated by the dynamical numerics (fig 15a).
%
% Heql  H_\infty points at which we plot the equilibrium solutions (fig 15a).
%
% Feql  Adhesion force in equilibrium at the points Heql (fig 15a).
%
% Rate  Different retraction speeds used (fig 15b - not ordered).
%
% W     Calculated energy to detach (fig 15b).
%
% Weql  Energy to 'yank' quasistatically, i.e. constant tension (fig 15b).
%
% Wtens Energy when peeling via tension change before 'yank' (fig 15b).

clear all;close all;


%% Load data
load('figure15a.mat')
load('figure15b.mat')

%% Plot 15a
% Plot dynamics
figure(1)
plot(H(:,1),F(:,1),'b-.','linewidth',2), hold on
plot(H(:,2),F(:,2),'r:','linewidth',2)

% Plot equilibria
plot([Heql(1:74); mean(Heql(74:75))],[Feql(1:74); mean(Feql(74:75))],...
     'k','linewidth',3,'handlevisibility','off')
plot([mean(Heql(74:75));Heql(75:105);mean(Heql(105:106))],...
    [mean(Feql(74:75));Feql(75:105);mean(Feql(105:106))],...
    'k--','linewidth',2,'handlevisibility','off')
plot([mean(Heql(105:106));Heql(106:end)],...
    [mean(Feql(105:106));Feql(106:end)],'k','linewidth',2)

% Add legend, axis labels, etc
leg = legend('$\dot{H}_\infty=1$', '$\dot{H_\infty}=50$', 'Equilibrium');
set(leg,'Interpreter','latex');

xlabel('$H_\infty$','interpreter','latex')
ylabel('$F$','interpreter','latex')
axis([5 20 0 1.2])

% Increase axis label and tick size
set(findobj('type','axes'),'fontsize',20,'linewidth',2)
set(gca,'TickLength',2*get(gca,'TickLength'))    

%% Plot 15b
% Plot dynamics
figure(2)
semilogx(Rate,W,'bx'), hold on

% Plot lines for equilibria
x = [1e-1 1e3];
plot(x,[Weql Weql],'k--')
plot(x,[Wtens Wtens],'r-.')

% Add legend, axis labels, etc
legend('Dynamic Yanking', 'Quasi-static Yanking', 'Tension change')
xlabel('$\dot{H}_\infty$','interpreter','latex')
ylabel('$W_{sep}$','interpreter','latex')
axis([1e-1 1e3 2 9])

% Increase axis label and tick size
set(findobj('type','axes'),'fontsize',20,'linewidth',2)
set(gca,'TickLength',2*get(gca,'TickLength'))    